const canvas = document.getElementById('game-canvas');
const ctx = canvas.getContext('2d');

const startScreen = document.getElementById('start-screen');
const gameOverScreen = document.getElementById('game-over-screen');
const startButton = document.getElementById('start-button');
const retryButton = document.getElementById('retry-button');
const titleButton = document.getElementById('title-button');

const currentScoreEl = document.getElementById('current-score');
const highScoreEl = document.getElementById('high-score');
const finalScoreEl = document.getElementById('final-score');

// Game variables
let snake, food, items, particles, score, highScore, animationFrameId;

// Game settings
const gridSize = 20; // The number of cells in the grid
let cellSize;
let snakeSize;
let speed;
const initialSpeed = 0.1; // Adjusted for grid-based speed

// Audio
const audioCtx = new (window.AudioContext || window.webkitAudioContext)();
const eatSound = new Audio('assets/eat.mp3');
// const gameOverSound = new Audio('assets/gameover.wav');

function resizeCanvas() {
    const screenWidth = window.innerWidth;
    const screenHeight = window.innerHeight;

    const size = Math.min(screenWidth, screenHeight);
    canvas.width = size;
    canvas.height = size;

    cellSize = canvas.width / gridSize;
    snakeSize = cellSize;
    speed = initialSpeed * cellSize;
}

window.addEventListener('resize', resizeCanvas);
resizeCanvas();

// Snake class
class Snake {
    constructor() {
        this.path = [{ x: Math.floor(gridSize / 2) * cellSize, y: Math.floor(gridSize / 2) * cellSize }];
        this.segments = 3;
        this.dx = speed;
        this.dy = 0;
    }

    draw() {
        ctx.fillStyle = 'green';
        for (let i = 0; i < this.segments; i++) {
            const index = this.path.length - 1 - i * (snakeSize / speed);
            if (index >= 0) {
                const point = this.path[Math.floor(index)];
                ctx.beginPath();
                ctx.arc(point.x, point.y, snakeSize / 2, 0, 2 * Math.PI);
                ctx.fill();
            }
        }
    }

    move() {
        this.dx = Math.sign(this.dx) * speed;
        this.dy = Math.sign(this.dy) * speed;
        const head = { x: this.path[this.path.length - 1].x + this.dx, y: this.path[this.path.length - 1].y + this.dy };
        this.path.push(head);
    }

    changeDirection(newDirection) {
        if (newDirection === 'up' && this.dy === 0) {
            this.dx = 0;
            this.dy = -speed;
        } else if (newDirection === 'down' && this.dy === 0) {
            this.dx = 0;
            this.dy = speed;
        } else if (newDirection === 'left' && this.dx === 0) {
            this.dx = -speed;
            this.dy = 0;
        } else if (newDirection === 'right' && this.dx === 0) {
            this.dx = speed;
            this.dy = 0;
        }
    }

    grow() {
        this.segments += 1;
    }

    checkCollision() {
        const head = this.getHead();

        // Wall collision
        if (head.x < 0 || head.x > canvas.width || head.y < 0 || head.y > canvas.height) {
            console.log(`Game Over: Wall collision. Head at (${head.x.toFixed(2)}, ${head.y.toFixed(2)})`);
            return true;
        }

        return false;
    }

    getHead() {
        return this.path[this.path.length - 1];
    }
}

// Food class
class Food {
    constructor() {
        this.size = cellSize / 2;
        this.place();
    }

    place() {
        this.x = (Math.floor(Math.random() * (gridSize - 4)) + 2 + 0.5) * cellSize;
        this.y = (Math.floor(Math.random() * (gridSize - 4)) + 2 + 0.5) * cellSize;
    }

    draw() {
        ctx.fillStyle = 'red';
        ctx.beginPath();
        ctx.arc(this.x, this.y, this.size, 0, 2 * Math.PI);
        ctx.fill();
    }
}

// Item class
class Item {
    constructor() {
        this.size = cellSize / 2;
        this.place();
    }

    place() {
        this.x = (Math.floor(Math.random() * (gridSize - 4)) + 2 + 0.5) * cellSize;
        this.y = (Math.floor(Math.random() * (gridSize - 4)) + 2 + 0.5) * cellSize;
    }

    draw() {
        // Implemented in subclasses
    }

    effect() {
        // Implemented in subclasses
    }
}

class SpeedUpItem extends Item {
    draw() {
        ctx.fillStyle = 'blue';
        ctx.beginPath();
        ctx.arc(this.x, this.y, this.size, 0, 2 * Math.PI);
        ctx.fill();
    }

    effect() {
        speed *= 1.5;
        setTimeout(() => {
            speed = initialSpeed * cellSize;
        }, 5000);
    }
}

class SpeedDownItem extends Item {
    draw() {
        ctx.fillStyle = 'orange';
        ctx.beginPath();
        ctx.arc(this.x, this.y, this.size, 0, 2 * Math.PI);
        ctx.fill();
    }

    effect() {
        speed /= 1.5;
        setTimeout(() => {
            speed = initialSpeed * cellSize;
        }, 5000);
    }
}

// Particle class
class Particle {
    constructor(x, y) {
        this.x = x;
        this.y = y;
        this.size = (Math.random() * 5 + 1) * (cellSize / 10);
        this.speedX = (Math.random() * 3 - 1.5) * (cellSize / 10);
        this.speedY = (Math.random() * 3 - 1.5) * (cellSize / 10);
        this.color = 'gold';
    }

    update() {
        this.x += this.speedX;
        this.y += this.speedY;
        if (this.size > 0.2 * (cellSize / 10)) this.size -= 0.1 * (cellSize / 10);
    }

    draw() {
        ctx.fillStyle = this.color;
        ctx.beginPath();
        ctx.arc(this.x, this.y, this.size, 0, 2 * Math.PI);
        ctx.fill();
    }
}

function createParticles(x, y) {
    for (let i = 0; i < 10; i++) {
        particles.push(new Particle(x, y));
    }
}

function handleParticles() {
    for (let i = 0; i < particles.length; i++) {
        particles[i].update();
        particles[i].draw();
        if (particles[i].size <= 0.2 * (cellSize / 10)) {
            particles.splice(i, 1);
            i--;
        }
    }
}

function playSound(sound) {
    sound.play();
}

function init() {
    resizeCanvas();
    snake = new Snake();
    food = new Food();
    items = [];
    particles = [];
    score = 0;
    highScore = localStorage.getItem('highScore') || 0;
    speed = initialSpeed * cellSize;

    currentScoreEl.textContent = score;
    highScoreEl.textContent = highScore;

    startScreen.style.display = 'none';
    gameOverScreen.style.display = 'none';

    mainLoop();
}

function mainLoop() {
    animationFrameId = requestAnimationFrame(mainLoop);

    speed += 0.0001 * cellSize;

    ctx.clearRect(0, 0, canvas.width, canvas.height);

    snake.move();
    snake.draw();
    food.draw();
    items.forEach(item => item.draw());
    handleParticles();

    if (snake.checkCollision()) {
        gameOver();
        return;
    }

    const head = snake.getHead();
    // Food collision
    const foodDistance = Math.sqrt(Math.pow(head.x - food.x, 2) + Math.pow(head.y - food.y, 2));
    if (foodDistance < snakeSize / 2 + food.size) {
        score += 10;
        currentScoreEl.textContent = score;
        snake.grow();
        createParticles(food.x, food.y);
        food.place();
        playSound(eatSound);
        // Add a new item every 5 foods eaten
        if (score % 50 === 0) {
            if (Math.random() > 0.5) {
                items.push(new SpeedUpItem());
            } else {
                items.push(new SpeedDownItem());
            }
        }
    }

    // Item collision
    items.forEach((item, index) => {
        const itemDistance = Math.sqrt(Math.pow(head.x - item.x, 2) + Math.pow(head.y - item.y, 2));
        if (itemDistance < snakeSize / 2 + item.size) {
            item.effect();
            items.splice(index, 1);
        }
    });
}

function gameOver() {
    cancelAnimationFrame(animationFrameId);
    // playSound(gameOverSound);
    if (score > highScore) {
        highScore = score;
        localStorage.setItem('highScore', highScore);
    }
    finalScoreEl.textContent = score;
    gameOverScreen.style.display = 'flex';
}

// Event Listeners
startButton.addEventListener('click', () => {
    startScreen.style.display = 'none';
    init();
});

retryButton.addEventListener('click', () => {
    gameOverScreen.style.display = 'none';
    init();
});

titleButton.addEventListener('click', () => {
    gameOverScreen.style.display = 'none';
    startScreen.style.display = 'flex';
});

// Keyboard controls
document.addEventListener('keydown', e => {
    const key = e.key;
    if (key === 'ArrowUp') {
        snake.changeDirection('up');
    } else if (key === 'ArrowDown') {
        snake.changeDirection('down');
    } else if (key === 'ArrowLeft') {
        snake.changeDirection('left');
    } else if (key === 'ArrowRight') {
        snake.changeDirection('right');
    }
});

// Swipe controls
let touchStartX = 0;
let touchStartY = 0;
let touchEndX = 0;
let touchEndY = 0;

canvas.addEventListener('touchstart', e => {
    touchStartX = e.changedTouches[0].screenX;
    touchStartY = e.changedTouches[0].screenY;
}, false);

canvas.addEventListener('touchend', e => {
    touchEndX = e.changedTouches[0].screenX;
    touchEndY = e.changedTouches[0].screenY;
    handleSwipe();
}, false);

function handleSwipe() {
    const dx = touchEndX - touchStartX;
    const dy = touchEndY - touchStartY;
    const angle = Math.atan2(dy, dx) * 180 / Math.PI;

    if (angle > -45 && angle <= 45) {
        snake.changeDirection('right');
    } else if (angle > 45 && angle <= 135) {
        snake.changeDirection('down');
    } else if (angle > 135 || angle <= -135) {
        snake.changeDirection('left');
    } else if (angle > -135 && angle <= -45) {
        snake.changeDirection('up');
    }
}

